"use client"

import { useState } from "react"
import { ChevronDown, ChevronUp, CheckCircle2 } from "lucide-react"
import { cn } from "@/lib/utils"
import type { FlowchartStage } from "@/lib/flowchart-data"

interface StageCardProps {
  stage: FlowchartStage
  isActive: boolean
  onClick: () => void
}

const colorClasses = {
  blue: {
    border: "border-primary/50",
    activeBorder: "border-primary",
    bg: "bg-primary/10",
    iconBg: "bg-primary/20",
    iconText: "text-primary",
    badge: "bg-primary text-primary-foreground",
  },
  green: {
    border: "border-accent/50",
    activeBorder: "border-accent",
    bg: "bg-accent/10",
    iconBg: "bg-accent/20",
    iconText: "text-accent",
    badge: "bg-accent text-accent-foreground",
  },
  amber: {
    border: "border-chart-3/50",
    activeBorder: "border-chart-3",
    bg: "bg-chart-3/10",
    iconBg: "bg-chart-3/20",
    iconText: "text-chart-3",
    badge: "bg-chart-3 text-background",
  },
  teal: {
    border: "border-chart-5/50",
    activeBorder: "border-chart-5",
    bg: "bg-chart-5/10",
    iconBg: "bg-chart-5/20",
    iconText: "text-chart-5",
    badge: "bg-chart-5 text-background",
  },
  rose: {
    border: "border-destructive/50",
    activeBorder: "border-destructive",
    bg: "bg-destructive/10",
    iconBg: "bg-destructive/20",
    iconText: "text-destructive",
    badge: "bg-destructive text-destructive-foreground",
  },
}

export function StageCard({ stage, isActive, onClick }: StageCardProps) {
  const [isExpanded, setIsExpanded] = useState(false)
  const colors = colorClasses[stage.color]
  const Icon = stage.icon

  return (
    <div
      className={cn(
        "relative rounded-lg border-2 bg-card transition-all duration-300 cursor-pointer",
        isActive ? colors.activeBorder : colors.border,
        isActive && "ring-2 ring-offset-2 ring-offset-background ring-primary/30"
      )}
      onClick={onClick}
    >
      {/* Stage Number Badge */}
      <div
        className={cn(
          "absolute -top-3 -left-3 w-8 h-8 rounded-full flex items-center justify-center text-sm font-bold",
          colors.badge
        )}
      >
        {stage.id}
      </div>

      {/* Header */}
      <div className={cn("p-4 rounded-t-lg", colors.bg)}>
        <div className="flex items-center gap-3">
          <div className={cn("p-2 rounded-lg", colors.iconBg)}>
            <Icon className={cn("w-5 h-5", colors.iconText)} />
          </div>
          <div className="flex-1 min-w-0">
            <h3 className="font-semibold text-foreground truncate">{stage.title}</h3>
            <p className="text-xs text-muted-foreground truncate">{stage.description}</p>
          </div>
        </div>
      </div>

      {/* Key Note */}
      {stage.keyNote && (
        <div className="px-4 py-2 bg-destructive/10 border-y border-destructive/20">
          <p className="text-xs font-medium text-destructive">{stage.keyNote}</p>
        </div>
      )}

      {/* Expandable Content */}
      <div className="p-4">
        <button
          type="button"
          onClick={(e) => {
            e.stopPropagation()
            setIsExpanded(!isExpanded)
          }}
          className="flex items-center gap-2 text-sm text-muted-foreground hover:text-foreground transition-colors w-full"
        >
          {isExpanded ? <ChevronUp className="w-4 h-4" /> : <ChevronDown className="w-4 h-4" />}
          <span>{isExpanded ? "Hide details" : "Show details"}</span>
        </button>

        {isExpanded && (
          <div className="mt-4 space-y-4 animate-in fade-in slide-in-from-top-2 duration-200">
            {/* Steps */}
            <div>
              <h4 className="text-xs font-semibold text-muted-foreground uppercase tracking-wider mb-2">
                Process Steps
              </h4>
              <ul className="space-y-2">
                {stage.steps.map((step, idx) => (
                  <li key={idx} className="flex items-start gap-2 text-sm text-foreground/80">
                    <span className={cn("w-5 h-5 rounded-full flex items-center justify-center text-xs flex-shrink-0 mt-0.5", colors.iconBg, colors.iconText)}>
                      {idx + 1}
                    </span>
                    <span>{step}</span>
                  </li>
                ))}
              </ul>
            </div>

            {/* Output */}
            <div>
              <h4 className="text-xs font-semibold text-muted-foreground uppercase tracking-wider mb-2">
                Output
              </h4>
              <div className="space-y-1">
                {stage.output.map((output, idx) => (
                  <div key={idx} className="flex items-center gap-2 text-sm">
                    <CheckCircle2 className="w-4 h-4 text-accent flex-shrink-0" />
                    <span className="text-foreground/90">{output}</span>
                  </div>
                ))}
              </div>
            </div>
          </div>
        )}
      </div>
    </div>
  )
}
